// Solace -- Sol Anachronistic Computer Emulation
// A Win32 emulator for the Sol-20 computer.
//
// Copyright (c) Jim Battle, 2001
//
// This file contains structures and function prototypes for
// manipulating virtual tapes for Solace, the Sol emulator.

#ifndef _VTAPE_H_
#define _VTAPE_H_

// -------- internal vtape respresentation --------
// the users of this interface shouldn't peer into this struct.

typedef unsigned short sample_t;

#define VT_LABELSIZE 2048
typedef struct {
    int      readonly;		// 1=write protected
    int      minutes;		// length of tape, in minutes
    int      maxsamples;	// length of tape, in samples
    int      dirty;		// has tape been modified?
    char     label[VT_LABELSIZE];	// ASCII label on tape cover
    int      ignorebaud;	// 1=don't worry about 300/1200
    int      lastsample;	// end of tape index -- one past last valid
    int      cursample;		// next sample to be read or written
    sample_t *sample;
} vtape_t;


// ------------ external values/functions ------------

// return status values
#define SVT_BADVALUE (-6)
#define SVT_BADPROP  (-5)
#define SVT_BADFILE  (-4)
#define SVT_BADTAPE  (-3)
#define SVT_EOF      (-2)
#define SVT_NOBYTE   (-1)
#define SVT_BYTE     ( 0)
#define SVT_OK       ( 0)
#define SVT_FRAMEERR ( 1)
#define SVT_OVERRUN  ( 2)
#define SVT_BOTHERR  ( 3)


// create pointer to a blank tape.
// returns NULL on error.
vtape_t *vtape_create(int minutes);

// destroy tape given pointer to it.
// caller is responsible for destroying pointer.
// returns 1 on error, 0 if OK.
int vtape_destroy(vtape_t *vt);

// initialize a tape
int vtape_init(vtape_t *vt);

// read file into supplied virtual tape
int vtape_fileread(vtape_t *vt, char *fname);

// write file from supplied virtual tape
int vtape_filewrite(vtape_t *vt, char *fname);

// get next "sample" off tape.  however, there might not be a character
// yet, or there might be errors.  we return one of the SVT_* status
// values.  curspeed is either 300 or 1200.
//int vtape_testbyte(vtape_t *vt, int curspeed, int *data);
int vtape_readbyte(vtape_t *vt, int curspeed, int *data, int *overrun, int *framing);

// write a byte to the current position on the tape.
// we return one of the SVT_* status codes.
//
// the flags parameter modifies what is written,
// and is one of the following SVT_* values:
//    SVT_NOBYTE, SVT_BYTE, SVT_FRAMEERR, SVT_OVERRUN, SVT_BOTHERR
//
// curspeed is either 300 or 1200.
//
// the return value is one of
//    SVT_BYTE    if OK
//    SVT_EOF     if end of tape
//    SVT_BADTAPE if bad tape foramt
int vtape_writebyte(vtape_t *vt, int curspeed, int data, int flags);

// get/set properties -- these are the properties that
// can be manipulated; BAUDRATE can't be set.
// POSITION is in 10ths of a second, which is 3 byte
// granularity at 300 baud, and 12 bytes at 1200 baud.
enum { VTPROP_POSITION=1, VTPROP_IGNOREBAUD=2, VTPROP_WRITEPROTECT=3,
       VTPROP_DIRTY=4, VTPROP_BAUDRATE=5, VTPROP_LABEL=6 };

// returns 0 on success, non-zero on failure.
int vtape_getprop(vtape_t *vt, int prop, int *value);
int vtape_setprop(vtape_t *vt, int prop, int  value);

#endif // ifdef _VTAPE_H_
